import Link from "next/link";
import { TopNav } from "@/components/nav";
import { Button, Card, CardBody, CardHeader, Container } from "@/components/ui";
import { requireSession } from "@/lib/session";
import { prisma } from "@/lib/prisma";
import { fromMinor } from "@/lib/money";

export default async function PosQrPage({ params }: { params: { token: string } }) {
  const session = await requireSession();
  const token = params.token;

  const merchant = await prisma.merchantProfile.findUnique({ where: { userId: session.uid } });
  if (!merchant) throw new Error("NOT_MERCHANT");

  const charge = await prisma.inStoreCharge.findUnique({ where: { token } });
  if (!charge || charge.merchantId !== merchant.id) throw new Error("NOT_FOUND");

  const baseUrl = process.env.NEXT_PUBLIC_BASE_URL ?? "";
  const payUrl = `${baseUrl}/pay/pos/${charge.token}`;

  return (
    <>
      <TopNav />
      <Container>
        <div className="mx-auto max-w-lg">
          <Card>
            <CardHeader title="Scan to pay" subtitle={`POS charge · ${charge.status}`} />
            <CardBody>
              <div className="text-sm text-zinc-600">Amount</div>
              <div className="mt-1 text-3xl font-semibold">P {fromMinor(charge.amount)}</div>
              <div className="mt-4 rounded-xl border border-zinc-200 bg-zinc-50 p-4">
                <div className="text-xs text-zinc-600">Customer link</div>
                <div className="mt-1 break-all text-sm font-medium">{payUrl}</div>
                <div className="mt-3">
                  <Link className="text-sm underline" href={`/pay/pos/${charge.token}`}>
                    Open customer pay page
                  </Link>
                </div>
              </div>

              <div className="mt-4 flex items-center gap-2">
                <Button asChild variant="secondary">
                  <Link href="/merchant/pos">Back to POS</Link>
                </Button>
                <Button asChild>
                  <Link href={`/merchant/pos/qr/${charge.token}/live`}>Live status</Link>
                </Button>
              </div>
              <div className="mt-3 text-xs text-zinc-500">
                Tip: keep this screen open; use Live status to watch the payment arrive.
              </div>
            </CardBody>
          </Card>
        </div>
      </Container>
    </>
  );
}
